package framework.everywaretechnologies.it.icarus;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.util.Log;
import android.view.accessibility.AccessibilityManager;

import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.LinkedList;
import java.util.Locale;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;


public class Icarus {

    //---------------- JSON KEYS ---------------------
    private final static String UTC_KEY="utc";
    private final static String USER_KEY="user";
    private final static String LANG_KEY="lang";
    private final static String ICARUS_VERSION_KEY="icarusversion";
    private final static String APP_NAME_KEY="appname";
    private final static String APP_VERSION_KEY="appversion";
    private final static String OS_VERSION_KEY="osversion";
    private final static String PLATFORM_KEY="platform";
    private final static String DEVICE_KEY="device";
    private final static String UUID_KEY="uuid";
    private final static String TALKBACK_KEY="talkback";
    private final static String DEBUG_KEY="debug";
    private final static String TIMESTAMP_KEY="timestamp";
    private final static String APP_DATA_KEY="appdata";
    private final static String USER_DATA_KEY="userdata";

    //------------------------------------------

    private static final Icarus ourInstance = new Icarus();
    private String icarusURL;
    private Queue<JSONObject> pendingResources; //FIFO
    private Lock pendingResourceLock;
    private Context appContext;// serve per istanziare un un'unica coda di richieste Volley (e non averne una per Activity)
    private static Context curContext; //context dell'Activity corrente
    private RequestQueue mVolleyRequestQueue;
    private SharedPreferences callerSharedPrefs;

    public static Icarus getInstance(Context c, String url) {

        curContext=c;
        ourInstance.icarusURL = url;
        return ourInstance;
    }

    private Icarus() {

        pendingResources=new LinkedList<>();
        pendingResourceLock=new ReentrantLock();
    }

    public void setApplicationContext(Context applicationContext){

        appContext=applicationContext;

        //adesso che ho l'applicationContext dell'app chiamante creo la coda di richieste per Volley
        mVolleyRequestQueue= Volley.newRequestQueue(appContext);
    }

    public void setCallerSharedPrefs(SharedPreferences callerSharedPrefs){

        this.callerSharedPrefs=callerSharedPrefs;
    }



    //----------------- LOG PREFERENCE CHANGES -----------------------

    public void logUserSettings(UserSettingsResource userSettingsResource, boolean debug){

        Log.d(curContext.getString(R.string.icarus_class),"log pref changes");

        JSONObject prefsChangesToLog=userSettingsResource.getUserSettingsChangesInJSONFormat();
        if(prefsChangesToLog.length()>0){ //se non ho un JSON vuoto, cioè qualche pref è cambiata

            postIcarusHttpRequest(prefsChangesToLog,debug);
            Log.d(curContext.getString(R.string.icarus_class),"log pref changes -> Volley");
        }
        userSettingsResource.clear();
    }

    //---------------------- LOG SCREEN TRACKING ------------------------------


    public void trackScreen(String screenEventType,boolean debug){

        Log.d(curContext.getString(R.string.icarus_class),"log screen track");

        String screenName=curContext.getClass().getSimpleName();

        if(screenEventType.equalsIgnoreCase(ScreenTrackResource.EVENT_ENTERED) || screenEventType.equalsIgnoreCase(ScreenTrackResource.EVENT_EXITED) ){

            IcarusResource resource=new ScreenTrackResource(screenName,screenEventType);

            Gson gson=new Gson();
            String screenTrackJson=gson.toJson(resource);
            try {

                JSONObject icarusPayload=new JSONObject(screenTrackJson);
                postIcarusHttpRequest(icarusPayload,debug);

                Log.d(curContext.getString(R.string.icarus_class),"log screen track -> Volley");

            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }


    //---------------------- LOG EVENT TRACKING --------------------------

    public void logEvent(EventTrackResource eventTrackResource, boolean debug){

        Log.d(curContext.getString(R.string.icarus_class),"log event -> Volley");

        JSONObject eventTrackJSON=eventTrackResource.getEventTrackResourceInJSONFormat();
        postIcarusHttpRequest(eventTrackJSON,debug);
    }

    //--------------------------- UTILITY ----------------------------------
    private void postIcarusHttpRequest(final JSONObject icarusPayload, final boolean debug){

        JSONObject toSendJson=createJSONToSend(icarusPayload,debug); //aggiunge informazioni al JSON ottenuto attraverso la raccolta dati dell'utente

        if(toSendJson==null)
            Log.e(curContext.getString(R.string.icarus_class),"call setCallerSharedPrefs(SharedPreferences prefs) on Icarus instance to do HTTP calls");
        else{

            final JsonObjectRequest jsonObjectRequest=new JsonObjectRequest(Request.Method.POST, ourInstance.icarusURL, toSendJson, new Response.Listener<JSONObject>() {
                @Override
                public void onResponse(JSONObject response) {

                    Log.d(curContext.getString(R.string.icarus_class),"onVolley response POSITIVE");

                    //controllo se ho qualche elemento che precedentemente non sono riuscito ad inviare e nel caso lo provo ad inviare
                    pendingResourceLock.lock();

                    if(pendingResources.peek() != null){ //se ho qualche elemento pendente da inviare

                        Log.d(curContext.getString(R.string.icarus_class),"messaggio pendente");
                        JSONObject pendingResource=pendingResources.poll();
                        postIcarusHttpRequest(pendingResource,debug);
                    }
                    pendingResourceLock.unlock();

                }
            }, new Response.ErrorListener() {
                @Override
                public void onErrorResponse(VolleyError error) {

                    Log.d(curContext.getString(R.string.icarus_class),"onVolley response NEGATIVE");

                    //se per qualche motivo la richiesta http non va a buon fine, salvo il payload che avrei dovuto mandare in cache
                    pendingResourceLock.lock();
                    pendingResources.add(icarusPayload);
                    pendingResourceLock.unlock();
                }
            });

            if(appContext!=null)
                mVolleyRequestQueue.add(jsonObjectRequest);//faccio effettivamente la richiesta
            else
                Log.e(curContext.getString(R.string.icarus_class),"call method setApplicationContext(Context appContext) on the Icarus instance to do HTTP calls");
        }
    }

    public JSONObject createJSONToSend(JSONObject userData, boolean debug){

        JSONObject toSend=new JSONObject();

        try {

            // ------------------- TIME JSON ----------------

            long utcTime=System.currentTimeMillis(); //ritorna il tempo UTC tra ora e il 1 gennaio 1970 alle 00:00
            TimeZone myTimeZone=TimeZone.getDefault();
            int timeOffsetBetweenMeAndGMT=myTimeZone.getOffset(utcTime); //ms
            long userTime=utcTime+timeOffsetBetweenMeAndGMT;

            JSONObject timeJSON=new JSONObject();
            timeJSON.put(UTC_KEY,utcTime);
            timeJSON.put(USER_KEY,userTime);

            //---------------------- APP JSON ---------------------

            //trovo il nome dell'app che sta chiamando questo metodo
            PackageManager pm=curContext.getPackageManager();
            CharSequence appName=pm.getApplicationLabel(curContext.getApplicationInfo());
            //trovo versione dell'app chiamante
            int clientAppVersion;
            try {

                PackageInfo pinfo=pm.getPackageInfo(curContext.getPackageName(),0);
                clientAppVersion=pinfo.versionCode;

            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();

                clientAppVersion=0;
            }
            //controllo se l'accessibilità è attivata o no
            AccessibilityManager am=(AccessibilityManager)curContext.getSystemService(Context.ACCESSIBILITY_SERVICE);

            boolean accessibilityEnabled=false;
            try {

                accessibilityEnabled=am.isTouchExplorationEnabled();

            } catch (NullPointerException npe){

                Log.e(curContext.getString(R.string.icarus_class),"errore ottenere accessibilityManager");
            }

            //controllo se nelle sharedPref dell'app chiamante è già presente un UUID, altrimenti ne creo uno che sarà costante per quell'utente
            if(callerSharedPrefs==null) return null;

            String uuid;
            if(callerSharedPrefs.contains("userUUID"))
                uuid=callerSharedPrefs.getString("userUUID","");
            else {
                uuid=UUID.randomUUID().toString();
                //update shared preferences dell'app chiamante
                SharedPreferences.Editor editor=callerSharedPrefs.edit();
                editor.putString("userUUID",uuid);
                editor.apply(); //apply aggiorna le prefs in bg
            }

            JSONObject appInfoJSON=new JSONObject();
            appInfoJSON.put(LANG_KEY, Locale.getDefault().getLanguage());
            appInfoJSON.put(ICARUS_VERSION_KEY,"Icarus:"+curContext.getResources().getInteger(R.integer.lib_version));
            appInfoJSON.put(APP_NAME_KEY,appName.toString());
            appInfoJSON.put(APP_VERSION_KEY,clientAppVersion);
            appInfoJSON.put(OS_VERSION_KEY, Build.VERSION.SDK_INT);
            appInfoJSON.put(PLATFORM_KEY,"android");
            appInfoJSON.put(DEVICE_KEY,getDeviceName());
            appInfoJSON.put(UUID_KEY,uuid);
            appInfoJSON.put(TALKBACK_KEY,accessibilityEnabled);

            //---------------- GENERAL JSON --------------------

            toSend.put(DEBUG_KEY,debug);
            toSend.put(TIMESTAMP_KEY,timeJSON);
            toSend.put(APP_DATA_KEY,appInfoJSON);
            toSend.put(USER_DATA_KEY,userData);

        } catch (JSONException e) {
            e.printStackTrace();
        }

        return toSend;
    }

    public String getDeviceName() {
        String manufacturer = Build.MANUFACTURER;
        String model = Build.MODEL;
        if (model.toLowerCase().startsWith(manufacturer.toLowerCase())) {
            return capitalize(model);
        } else {
            return capitalize(manufacturer) + " " + model;
        }
    }


    private String capitalize(String s) {
        if (s == null || s.length() == 0) {
            return "";
        }
        char first = s.charAt(0);
        if (Character.isUpperCase(first)) {
            return s;
        } else {
            return Character.toUpperCase(first) + s.substring(1);
        }
    }
}
